import sys

VALID_SYMBOLS = {".", "#", "*"}

def validate_world(path="world_002.txt", rows=10, cols=10):
    """
    Validate world.txt according to the rules of the assignment.

    Returns:
        (ok, errors)
        ok: True/False
        errors: list of error-messages
    """

    errors = []

    # ----------------------------
    # STEP 1 – Διαβάζουμε το αρχείο raw
    # ----------------------------
    try:
        with open(path, "r", encoding="utf-8") as f:
            raw_lines = f.readlines()
    except UnicodeDecodeError:
        errors.append(" Encoding error: Το αρχείο ΔΕΝ είναι UTF-8.")
        return False, errors
    except FileNotFoundError:
        errors.append(" Το αρχείο world.txt δεν βρέθηκε.")
        return False, errors

    # ----------------------------
    # STEP 2 – Αφαίρεση newlines
    # ----------------------------
    lines = [line.rstrip("\n") for line in raw_lines]

    # ----------------------------
    # STEP 3 – Έλεγχος για BOM
    # ----------------------------
    if lines and lines[0].startswith("\ufeff"):
        errors.append(" Εντοπίστηκε BOM (\\ufeff) στην πρώτη γραμμή. \
Αποθηκεύστε ως UTF-8 χωρίς BOM.")
        # καθαρίζουμε για να συνεχίσει ο validator
        lines[0] = lines[0].replace("\ufeff", "")

    # ----------------------------
    # STEP 4 – Αφαίρεση κενών γραμμών
    # ----------------------------
    nonempty_lines = [line for line in lines if line.strip() != ""]

    if len(nonempty_lines) != rows:
        errors.append(f" Λάθος αριθμός γραμμών: Βρέθηκαν {len(nonempty_lines)}, \
απαιτούνται ακριβώς {rows}.")

    # ----------------------------
    # STEP 5 – Split κάθε γραμμή σε tokens
    # ----------------------------
    world = []
    for idx, line in enumerate(nonempty_lines):
        tokens = line.strip().split()
        if len(tokens) != cols:
            errors.append(
                f" Στη γραμμή {idx+1}: Βρέθηκαν {len(tokens)} tokens, απαιτούνται {cols}."
            )
        world.append(tokens)

    # Αν δεν έχουμε ακριβώς 10x10, σταματάμε
    if errors:
        return False, errors

    # ----------------------------
    # STEP 6 – Έλεγχος συμβόλων
    # ----------------------------
    for r in range(rows):
        for c in range(cols):
            cell = world[r][c]
            if cell not in VALID_SYMBOLS:
                errors.append(
                    f" Άκυρο σύμβολο στη θέση ({r},{c}): '{cell}'. "
                    "Επιτρέπονται μόνο '.', '#', '*'."
                )

    # ----------------------------
    # STEP 7 – Έλεγχος κρυφών unicode
    # ----------------------------
    for r in range(rows):
        for c in range(cols):
            cell = world[r][c]
            for ch in cell:
                if ord(ch) > 127:
                    errors.append(
                        f" Κρυφός unicode χαρακτήρας στη θέση ({r},{c}): repr={repr(ch)}"
                    )

    # ----------------------------
    # STEP 8 – Προαιρετικοί λογικοί έλεγχοι
    # (Δεν είναι υποχρεωτικοί στην εργασία, αλλά βοηθούν στο debugging)
    # ----------------------------

    # 8a: Έλεγχος unreachable blocks (όχι BFS, απλός heuristic)
    items = [(r,c) for r in range(rows) for c in range(cols) if world[r][c] == "*"]
    if not items:
        errors.append(" Προειδοποίηση: Δεν βρέθηκαν αντικείμενα '*' στο world.txt.")

    # 8b: Έλεγχος αν το (0,0) είναι εμπόδιο
    if world[0][0] == "#":
        errors.append(" Προειδοποίηση: Το start (0,0) είναι εμπόδιο '#'. "
                      "Το ρομπότ δεν θα μπορεί να μετακινηθεί.")

    # ----------------------------
    # ΤΕΛΙΚΟ ΑΠΟΤΕΛΕΣΜΑ
    # ----------------------------
    if errors:
        return False, errors
    return True, [" Το world.txt είναι έγκυρο."]


# -----------------------------------------
# Εκτελέσιμο μέρος – επιτρέπει CLI χρήση
# -----------------------------------------
if __name__ == "__main__":
    ok, msgs = validate_world()
    for m in msgs:
        print(m)
    if not ok:
        sys.exit(1)
